/*
============================================================
 The following is information regarding a single item from:
 Resource: "patient-facilities"
============================================================
 Body: {
   "institution": {
     "institutionCode": "631GC",
     "name": "PITTSFIELD CBOC",
     "city": "PITTSFIELD",
     "stateAbbrev": "MA"
   },
   "institutionTimezone": "America/New_York",
   "requestSupported": false,
   "directSchedulingSupported": true
 }

============================================================
 Schema: {
   "type": "object",
   "properties": {
     "institution": {
       "type": "object",
       "properties": {
         "institutionCode": {
           "type": "string"
         },
         "name": {
           "type": "string"
         },
         "city": {
           "type": "string"
         },
         "stateAbbrev": {
           "type": "string"
         }
       }
     },
     "institutionTimezone": {
         "type": "string"
     },
     "requestSupported": {
       "type": "boolean"
     },
     "directSchedulingSupported": {
       "type": "boolean"
     }
   }
 }
 ============================================================
 */
define([
    'underscore',
    'backbone',
    'App',
    'modules/new-appointment-request/resources/common/facility/express-time-model',
], function(_, Backbone, app, TimeModel) {
    'use strict';

    return Backbone.Model.extend({
        idAttribute: 'institutionCode',

        // eslint-disable-next-line complexity
        parse: function(response) {
            var _response = response || {};
            var attributes = _.extend({}, _response);

            if (_response.institution) {
                attributes = _.omit(attributes, 'institution');
                attributes = _.extend(attributes, _response.institution);
            }

            if (_response.institutionTimezone) {
                attributes = _.extend(attributes, _response.institutionTimezone);
            }

            if (!_.isBoolean(attributes.requestSupported)) {
                attributes.requestSupported = Boolean(_response.requestSupported);
            }

            if (!_.isBoolean(attributes.directSchedulingSupported)) {
                attributes.directSchedulingSupported = Boolean(_response.directSchedulingSupported);
            }

            if (_.isObject(attributes.expressTimes)) {
                attributes.expressTimes = new TimeModel(attributes.expressTimes);
            }

            return attributes;
        },

        getExpressMessage: function() {
            var expressTime = this.get('expressTimes');
            return expressTime.getExpressMessage();
        },

        isExpressSupportedToday: function() {
            var expressTime = this.get('expressTimes');
            return _.isObject(expressTime);
        },

        isExpressSupportedAtCurrentTime: function() {
            var expressTime = this.get('expressTimes');
            return expressTime.isExpressAvailable();
        },

        /**
         * @return {boolean}
         */
        isRequestSupported: function() {
            return this.get('requestSupported');
        },

        /**
         * @return {boolean}
         */
        isDirectSchedulingSupported: function() {
            return this.get('directSchedulingSupported');
        },

        /**
         * Maps the model to a selection view value
         * @param {array} clinicsInstitutionCodes the list of institution codes
         * @return {{id: string, value: string, text: string}}
         */
        toSectionData: function(clinicsInstitutionCodes) {
            var hasCode = clinicsInstitutionCodes ?
                clinicsInstitutionCodes.indexOf(this.get('institutionCode')) !== -1
                : false;
            var printPactLabel = hasCode && _.isUndefined(app.disabledFeatures.get('PACT_LOCATION_FEATURE_SET'));

            return {
                id: this.id,
                value: this.id,
                hasPactLabel: printPactLabel,
                text: this.formatTextValue(printPactLabel),
            };
        },

        /**
         * Get the text value for the dropdown entry corresponding to this model object
         * @param {boolean} printPactLabel Print the PACT notification
         * @returns {string} The label
         */
        formatTextValue: function(printPactLabel) {
            var name = this.get('name');
            var city = this.get('city');
            var state = this.get('stateAbbrev');
            var pacTeamLabel = '';
            var cityAndStateLabel = (city && state) ?
                ' (' + city + ', ' + state + ')' :
                '';
            if (printPactLabel) {
                pacTeamLabel = ' - Your Primary Care Team';
            }

            return name + cityAndStateLabel + pacTeamLabel;
        },
    });
});
